#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>
#include "thpmd.h"

typedef struct {
	u16 entrysize;
/*	u16 unk_1; */
	u16 entrynum;
	u8 entrykey;
/*	u8 padding[9]; */
} th_dat_header;

typedef struct {
	u16 magic;
	u8  key;
	u8  name[13];
	u16 datalen;
	u16 filelen;
	u32 offset;
/*	u8  padding[8]; */
} th_file_entry;

int th_unrle(const u8 *in, u32 insize, u8 *out, u32 outsize)
{
	u8 ppr, pr, i;
	u32 inpos, outpos;

	pr = out[0] = in[0];
	for(inpos = outpos = 1; inpos < insize && outpos < outsize;) {
		ppr = pr;
		pr = out[outpos++] = in[inpos++];
		if(ppr == pr) {
			for (i = 0; i < in[inpos] && outpos < outsize; ++i)
				out[outpos++] = pr;
			++inpos;
		}
	}
	return (inpos == insize);
}

int th_extract(FILE *fi, th_file_entry *entry)
{
	FILE *fo;
	char *file;
	int i;

	printf("Extracting %s...\n", entry->name);

	if(fseek(fi, entry->offset, SEEK_SET) == -1) {
		perror("fseek");
		return 1;
	}

	file = (char *)malloc(entry->datalen);
	if(!file) {
		perror("malloc");
		return 1;
	}
	
	if(fread(file, 1, entry->datalen, fi) != entry->datalen) {
		if(feof(fi)) {
			fprintf(stderr, "THPMD: Data seems incomplete\n");
		} else {
			perror("fread");
		}
		free(file);
		return 1;
	}

	/* f[^𕜌 */
	if (entry->key != 0) {
		for(i = 0; i < entry->datalen; ++i) file[i] ^= entry->key;
	}

	/* KvɉRLEWJ */
	if(entry->datalen < entry->filelen) {
		char *plain = malloc(entry->filelen);
		if(!plain) {
			perror("malloc");
			free(file);
			return 0;
		}
		if(!th_unrle(file, entry->datalen, plain, entry->filelen)) {
			fprintf(stderr, "THPMD: Failed to uncompress data");
			free(file);
			free(plain);
			return 0;
		}
		free(file);
		file = plain;
	}

	fo = fopen(entry->name, "wb");
	if(fo == NULL) {
		perror("fopen");
		free(file);
		return 1;
	}
	if(fwrite(file, 1, entry->filelen, fo) != entry->filelen) {
		perror("fwrite");
		fclose(fo);
		free(file);
		return 1;
	}

	fclose(fo);
	free(file);
	
	return 0;
}
int th_readheader(FILE *fp, th_dat_header *hed)
{
	u8 buf[0x10];
	if(fread(buf, 1, 0x10, fp) != 0x10) return 0;
	hed->entrysize = to16le(buf);
	hed->entrynum = to16le(buf + 4);
	hed->entrykey = *(buf + 6);
	return 1;
}

int th_chkheader(const th_dat_header *hed, u32 filelen)
{
	if(hed->entrysize > filelen) return 0;
	if(hed->entrysize & 0x1F || hed->entrysize / 0x20 < hed->entrynum) return 0;
	return 1;
}

void th_copyentry(const u8 *buf, th_file_entry *entry)
{
	entry->magic = to16le(buf);
	entry->key = *(buf + 2);
	memcpy(entry->name, buf + 3, 13);
	entry->datalen = to32le(buf + 0x10);
	entry->filelen = to32le(buf + 0x12);
	entry->offset = to32le(buf + 0x14);
}

int th_chkentry(const th_file_entry *entry, u32 filelen)
{
	if(entry->magic != 0x9595 && entry->magic != 0xF388) return 0;
	if(entry->offset >= filelen) return 0;
	if(entry->datalen == 0 || filelen - entry->offset < entry->datalen) return 0;
	return 1;
}

int th_chkname(const char *name)
{
	int i, j;
	/* t@C8.3`Ȃƃ_ */
	for(i = j = 0; i < 8 && isfchr(name[i]); ++i);
	if(name[i] == '.')
		for(j = 1; j < 4 && isfchr(name[i + j]); ++j);
	return (name[i + j] == '\0') && i && (j != 1);
}

int th_ispmd(const char *name)
{
	/* ̑ */
	char *ext = strchr(name, '.');
	if(!ext) return 0;
	return
		!strcmp(++ext, "M") || !strcmp(ext, "M2") ||
		!strcmp(ext, "M26") || !strcmp(ext, "M86") ||
		!strcmp(ext, "MMD");
}

void usage()
{
	puts(
	"THPMD for TH03, TH04 and TH05.\n"
	"\n"
	"Usage: THPMDC FILE"
	);
}

int main(int argc, char *argv[])
{
	FILE *fi;
	long filelen;
	th_dat_header hed;
	th_file_entry entry;
	u8 *entrybuf, *p;
	int i;

	if(argc < 2) {
		usage();
		return EXIT_SUCCESS;
	}

	fi = fopen(argv[1], "rb");
	if(!fi) {
		perror("fopen");
		return EXIT_FAILURE;
	}

	fseek(fi, 0, SEEK_END);
	filelen = ftell(fi);
	rewind(fi);

	/* wb_ǂ */
	if(!th_readheader(fi, &hed)) {
		if(feof(fi)) {
			fprintf(stderr, "Input file too small.");
		} else {
			perror("fread");
		}
		fclose(fi);
		return EXIT_FAILURE;
	}

	/* wb_̐ */
	if(!th_chkheader(&hed, filelen)) {
		fprintf(stderr, "THPMD: Header seems broken.");
		fclose(fi);
		return EXIT_FAILURE;
	}

	/* t@CGgǂ */
	entrybuf = malloc(hed.entrysize);
	if(!entrybuf) {
		perror("malloc");
		fclose(fi);
		return EXIT_FAILURE;
	}
	if(fread(entrybuf, 1, hed.entrysize, fi) != hed.entrysize) {
		if(feof(fi)) {
			fprintf(stderr, "THPMD: File entries seem incomplete.");
		} else {
			perror("fread");
		}
		free(entrybuf);
		fclose(fi);
		return EXIT_FAILURE;
	}

	/* t@CGg𕜌 */
	for(i = 0; i < hed.entrysize; ++i) {
		entrybuf[i] ^= hed.entrykey;
		hed.entrykey -= entrybuf[i];
	}

	p = entrybuf;
	for(i = 0; i < hed.entrynum; ++i, p += 0x20) {
		th_copyentry(p, &entry);
		/* t@CGg̐ */
		if(entry.magic == 0) break;
		if(!th_chkentry(&entry, filelen) || 
		   !th_chkname(entry.name)) {
			fprintf(stderr, "THPMD: File entry seems broken.");
			free(entrybuf);
			fclose(fi);
			return EXIT_FAILURE;
		}

		if(th_ispmd(entry.name)) {
			if(th_extract(fi, &entry)) {
				free(entrybuf);
				fclose(fi);
				return EXIT_FAILURE;
			}
		}
	}

	puts("Finished extracting files.");

	free(entrybuf);
	fclose(fi);

	return EXIT_SUCCESS;
}
